<?php
/*--------------------------------------------------------------------------------------------------
    OnGetSelectedQuantityEventListener.php 2020-03-18
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2016 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */

namespace Gambio\Shop\Product\SellingUnitQuantitiy\Database\Listeners;

use Gambio\Shop\SellingUnit\Unit\Exceptions\InsufficientQuantityException;
use Gambio\Shop\SellingUnit\Unit\Exceptions\InvalidQuantityGranularityException;
use Gambio\Shop\SellingUnit\Unit\Exceptions\QuantitySurpassMaximumAllowedQuantityException;
use Gambio\Shop\SellingUnit\Unit\Exceptions\RequestedQuantityBelowMinimumException;
use Gambio\Shop\SellingUnit\Database\Unit\Events\Interfaces\OnGetSelectedQuantityEventInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Interfaces\IncludeReservedOnQuantityInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitId;

/**
 * Class OnGetSelectedQuantityListener
 */
class OnGetSelectedQuantityEventListener
{
    /**
     * @var float|double
     */
    private $maxProductQuantity;


    /**
     * OnGetSelectedQuantityListener constructor.
     *
     * @param float $maxProductQuantity
     */
    public function __construct(float $maxProductQuantity)
    {

        $this->maxProductQuantity = $maxProductQuantity;
    }


    /**
     * @param OnGetSelectedQuantityEventInterface $event
     *
     * @return OnGetSelectedQuantityEventInterface
     */
    public function __invoke(OnGetSelectedQuantityEventInterface $event): OnGetSelectedQuantityEventInterface
    {
        $quantity = $event->requestedQuantity()->value();

        if ($event->requestedQuantity() instanceof IncludeReservedOnQuantityInterface) {
            $quantity += $this->getReservedQuantity($event->id());
        }


        if ($quantity > $this->maxProductQuantity) {
            $event->requestedQuantity()->stackException(new QuantitySurpassMaximumAllowedQuantityException($event->id()
                ->productId()
                ->value(),
                $event->requestedQuantity()
                    ->value(),
                $this->maxProductQuantity,
                $event->requestedQuantity()->exception()));
        }

        if ($quantity < $event->product()->getMinOrder()) {
            $event->requestedQuantity()->stackException(new RequestedQuantityBelowMinimumException($event->id()->productId()->value(),
                $quantity,
                $event->product()->getMinOrder(),
                $event->requestedQuantity()->exception()));
        }

        $mod = $quantity / $event->product()->getGranularity();
        $mod = ($mod - (float)$mod) * $event->product()->getGranularity();
        if ($mod > 0) {
            $event->requestedQuantity()->stackException(new InvalidQuantityGranularityException($event->id()->productId()->value(),
                $event->quantityGraduation()->value(),
                $event->requestedQuantity()->exception()));
        }

        if ($quantity > $event->stock()->availableQuantity()->value()) {
            $event->requestedQuantity()->stackException(new InsufficientQuantityException($event->id()->productId()->value(), $event->stock()->availableQuantity()->value(), $event->requestedQuantity()->exception()));
        }

        $event->setSelectedQuantity($event->requestedQuantity());

        return $event;
    }

    /**
     * @param SellingUnitId $id
     * @return float
     */
    protected function getReservedQuantity(SellingUnitId $id): float
    {
        $result = 0;
        if (isset($_SESSION['cart']) && is_object($_SESSION['cart'])) {

            $result = (float)$_SESSION['cart']->get_quantity($id->productId()->value());
        }

        return $result;
    }
}